/* $ZEL: sis1100_write_loop.c,v 1.2 2004/05/27 23:10:40 wuestner Exp $ */

/*
 * Copyright (c) 2001-2004
 * 	Peter Wuestner.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "sis1100_sc.h"

#if !defined(__NetBSD__) && ! defined(__linux__)
#error Invalid or Unknown Operating System
#endif

/*
 * sis1100_write_loop always transports data directly from userspace!
 * Access permissions have to be checked before.
 */

ssize_t
sis1100_write_loop(
    struct sis1100_softc* sc,
    struct sis1100_fdata* fd,
    u_int32_t addr,           /* VME or SDRAM address */
    int32_t am,               /* address modifier, not used if <0 */
    int size,                 /* datasize */
    int space,                /* remote space (1,2: VME; 6: SDRAM) */
    int fifo_mode,
    size_t count,             /* words (of size 'size') to be transferred */
                              /* count==0 is illegal */
    size_t* count_written,    /* words transferred */
    const u_int8_t* data,     /* source (user virtual address) */
    int* prot_error
    )
{
    u_int32_t head;
    int idx;

    *count_written=count;
    head=0x0f000404|(space&0x3f)<<16;
    SEM_LOCK(sc->sem_hw);
    if (am>=0) {
        head|=0x800;
        sis1100writereg(sc, t_am, am);
    }
    switch (size) {
    case 1:
        if (fifo_mode) {
            sis1100writereg(sc, t_adl, addr);
            sis1100writereg(sc, t_hdr, head|(0x01000000<<(addr&3)));
            for (idx=0; idx<count; idx++, data++) {
                u_int32_t val;
#ifdef __NetBSD__
                val=fubyte(data);
#elif __linux__
                __get_user(val, (u_int8_t*)data);
#endif
                val=(val&0xff)<<((addr&3)<<3);
                sis1100rawwritereg(sc, t_dal, val);
            }
        } else {
            for (idx=0; idx<count; idx++, data++, addr++) {
                u_int32_t val;
#ifdef __NetBSD__
                val=fubyte(data);
#elif __linux__
                __get_user(val, (u_int8_t*)data);
#endif
                val=(val&0xff)<<((addr&3)<<3);
                sis1100writereg(sc, t_hdr, head|(0x01000000<<(addr&3)));
                sis1100writereg(sc, t_adl, addr);
                sis1100rawwritereg(sc, t_dal, val);
            }
        }
        break;
    case 2:
        if (fifo_mode) {
            sis1100writereg(sc, t_adl, addr);
            sis1100writereg(sc, t_hdr, head|(0x03000000<<(addr&3)));
            for (idx=0; idx<count; idx++, data+=2) {
                u_int32_t val;
#ifdef __NetBSD__
                val=fusword(data);
#elif __linux__
                __get_user(val, (u_int16_t*)data);
#endif
                val=(val&0xffff)<<((addr&3)<<3);
                sis1100rawwritereg(sc, t_dal, val);
            }
        } else {
            for (idx=0; idx<count; idx++, data+=2, addr+=2) {
                u_int32_t val;
#ifdef __NetBSD__
                val=fusword(data);
#elif __linux__
                __get_user(val, (u_int16_t*)data);
#endif
                val=(val&0xffff)<<((addr&3)<<3);
                sis1100writereg(sc, t_hdr, head|(0x03000000<<(addr&3)));
                sis1100writereg(sc, t_adl, addr);
                sis1100rawwritereg(sc, t_dal, val);
            }
        }
        break;
    case 4:
        sis1100writereg(sc, t_hdr, head|0x0f000000);
        if (fifo_mode) {
            sis1100writereg(sc, t_adl, addr);
            for (idx=0; idx<count; idx++, data+=4) {
                u_int32_t val;
#ifdef __NetBSD__
                val=fuword(data);
#elif __linux__
                __get_user(val, (u_int32_t*)data);
#endif
                sis1100rawwritereg(sc, t_dal, val);
            }
        } else {
            for (idx=0; idx<count; idx++, data+=4, addr+=4) {
                u_int32_t val;
#ifdef __NetBSD__
                val=fuword(data);
#elif __linux__
                __get_user(val, (u_int32_t*)data);
#endif
                sis1100writereg(sc, t_adl, addr);
                sis1100writereg(sc, t_dal, val);
            }
        }
        break;
    }

    do {
        *prot_error=sis1100readreg(sc, prot_error);
    } while (*prot_error==0x005);

    SEM_UNLOCK(sc->sem_hw);

    /*
    This is not really correct, but we don't know how many data
    have been successfully written before an error occured.
    */
    return *prot_error?EIO:0;
}
