/* $ZEL: sis1100_read_loop.c,v 1.4 2005/07/07 14:15:38 wuestner Exp $ */

/*
 * Copyright (c) 2003-2004
 * 	Peter Wuestner.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "sis1100_sc.h"

/*
 * sis1100_read_loop always transports data directly to userspace!
 * Access permissions have to be checked before.
 */

int
sis1100_read_loop(
    struct sis1100_softc* sc,
    struct sis1100_fdata* fd,
    u_int32_t addr,           /* VME or SDRAM address */
    int32_t am,               /* address modifier, not used if <0 */
    int size,                 /* datasize (bytes/word) */
    int space,                /* remote space (1,2: VME; 6: SDRAM) */
    int fifo_mode,
    size_t count,             /* words (of size 'size') to be transferred */
                              /* count==0 is illegal */
    size_t* count_read,       /* words transferred */
    u_int8_t* data,           /* destination (user virtual address) */
    int* prot_error
    )
{
    u_int32_t head;
    int idx, res=0;

    *count_read=count;
    head=0x00000002|(space&0x3f)<<16;
    SEM_LOCK(sc->sem_hw);
    if (am>=0) {
        head|=0x800;
        sis1100writereg(sc, t_am, am);
    }
    switch (size) {
    case 1:
        for (idx=0; idx<count; idx++, data++) {
            u_int32_t val;
            sis1100writereg(sc, t_hdr, head|(0x01000000<<(addr&3)));
            sis1100writereg(sc, t_adl, addr);
            do {
                *prot_error=sis1100readreg(sc, prot_error);
            } while (*prot_error==0x005);
            if (*prot_error) {
                *count_read=idx;
                break;
            }
            val=sis1100rawreadreg(sc, tc_dal);
#ifdef __NetBSD__
#if defined(__LITTLE_ENDIAN)
            subyte(data, (val>>((addr&3)<<3))&0xff);
#else
            subyte(data, (val>>((3-(addr&3))<<3))&0xff);
#endif
#elif __linux__
#if defined(__LITTLE_ENDIAN)
            __put_user((val>>((addr&3)<<3))&0xff, (u_int8_t*)(data));
#else
            __put_user((val>>((3-(addr&3))<<3))&0xff, (u_int8_t*)(data));
#endif
#endif
            if (!fifo_mode) addr++;
        }
        break;
    case 2:
        for (idx=0; idx<count; idx++, data+=2) {
            u_int32_t val;
            sis1100writereg(sc, t_hdr, head|(0x03000000<<(addr&3)));
            sis1100writereg(sc, t_adl, addr);
            do {
                *prot_error=sis1100readreg(sc, prot_error);
            } while (*prot_error==0x005);
            if (*prot_error) {
                /*
                pINFO(sc, "read_loop(2): addr=%x prot_error=%x idx=%d",
                    addr, *prot_error, idx);
                */
                *count_read=idx;
                break;
            }
            val=sis1100rawreadreg(sc, tc_dal);

    pERROR(sc, "read_loop: head=%08x addr=%x idx=%d val=%08x",
        head|(0x03000000<<(addr&3)), addr, idx, val);

#ifdef __NetBSD__
#if defined(__LITTLE_ENDIAN)
            susword(data, (val>>((addr&2)<<3))&0xffff);
#else
            susword(data, (val>>((2-(addr&2))<<3))&0xffff);
#endif
#elif __linux__
#if defined(__LITTLE_ENDIAN)
            __put_user((val>>((addr&2)<<3))&0xffff, (u_int16_t*)data);
#else
            __put_user((val>>((2-(addr&2))<<3))&0xffff, (u_int16_t*)data);
#endif
#endif
            if (!fifo_mode) addr+=2;
        }
        break;
    case 4:
        sis1100writereg(sc, t_hdr, head|0x0f000000);
        for (idx=0; idx<count; idx++, data+=4) {
            u_int32_t val;
            sis1100writereg(sc, t_adl, addr);
            do {
                *prot_error=sis1100readreg(sc, prot_error);
            } while (*prot_error==0x005);
            if (*prot_error) {
                *count_read=idx;
                break;
            }
            val=sis1100rawreadreg(sc, tc_dal);
#ifdef __NetBSD__
            suword(data, val);
#elif __linux__
            __put_user(val, (u_int32_t*)data);
#endif
            if (!fifo_mode) addr+=4;
        }
        break;
    }
    SEM_UNLOCK(sc->sem_hw);
/*
pINFO(sc, "read_loop(am=%x size=%d count=%d count_read=%d, addr=%x): res=%d",
    am, size, count, *count_read, addr, res);
*/
    return res;
}
