/* $ZEL: sis1100_irq.c,v 1.10 2006/02/14 19:51:19 wuestner Exp $ */

/*
 * Copyright (c) 2001-2004
 * 	Matthias Drochner, Peter Wuestner.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "sis1100_sc.h"

int
sis1100_enable_irq(struct sis1100_softc* sc,
    u_int32_t plx_mask, u_int32_t sis_mask)
{
    DECLARE_SPINLOCKFLAGS(flags)
    if (plx_mask) {
        SPIN_LOCK_IRQSAVE(sc->lock_intcsr, flags);
        plxwritereg(sc, INTCSR, plxreadreg(sc, INTCSR)|plx_mask);
        SPIN_UNLOCK_IRQRESTORE(sc->lock_intcsr, flags);
    }

    if (sis_mask) {
        sis_mask&=sis1100_all_irq;
        sis1100writereg(sc, sr, sis_mask); /* clear pending irqs */
        sis1100writereg(sc, cr, sis_mask); /* enable irqs */
    }
    return 0;
}

int
sis1100_disable_irq(struct sis1100_softc* sc,
    u_int32_t plx_mask, u_int32_t sis_mask)
{
    DECLARE_SPINLOCKFLAGS(flags)
    if (plx_mask) {
        SPIN_LOCK_IRQSAVE(sc->lock_intcsr, flags);
        plxwritereg(sc, INTCSR, plxreadreg(sc, INTCSR)&~plx_mask);
        SPIN_UNLOCK_IRQRESTORE(sc->lock_intcsr, flags);
    }

    if (sis_mask) sis1100writereg(sc, cr, (sis_mask&sis1100_all_irq)<<16);
    return 0;
}

/* Doorbell | Local | DMA0 | DMA1 */
#define HANDLED_IRQS (plxirq_doorbell_active|plxirq_local_active|\
                      plxirq_dma0_active|plxirq_dma1_active)

#define PLXIRQ_BRAKE 1000
#undef PLXIRQ_BRAKE
#undef PLXIRQ_DEBUG

#ifdef PLXIRQ_DEBUG
#define pPLXIRQ(sc, fmt, arg...) pERROR(sc, fmt, ## arg)
#else
#define pPLXIRQ(sc, fmt, arg...) do {} while(0)
#endif

#ifdef __NetBSD__
int
sis1100_intr(void* vsc)
#elif __linux__
irqreturn_t
sis1100_intr(int irq, void *vsc, struct pt_regs *regs)
#endif
{
    DECLARE_SPINLOCKFLAGS(flags)
    struct sis1100_softc* sc=(struct sis1100_softc*)vsc;
    u_int32_t intcsr;
    int local, handler_command, wakeup_handler, wakeup_local;
#ifdef PLXIRQ_BRAKE
    static int emergency_count=0;
#endif
    u_int32_t rest;

    intcsr=plxreadreg(sc, INTCSR);
    if (!(intcsr & HANDLED_IRQS)) {
        pPLXIRQ(sc, "NIX IRQ! intcsr=0x%x", intcsr);
        return IRQ_NONE;
    } else {
        pPLXIRQ(sc, "HAB IRQ! intcsr=0x%x", intcsr);
    }

#ifdef PLXIRQ_BRAKE
    if (++emergency_count>PLXIRQ_BRAKE) {
        sis1100_disable_irq(sc, plxirq_pci, 0);
        pERROR(sc, "irqs disabled after %d", PLXIRQ_BRAKE);
    }
#endif

    local=0;
    handler_command=0;
    wakeup_handler=0;
    wakeup_local=0;

    if (intcsr&plxirq_doorbell_active) { /* == VME/CAMAC IRQ */
        u_int32_t help=plxreadreg(sc, L2PDBELL);
        pPLXIRQ(sc, "plxirq: doorbell=0x%x", help);
        SPIN_LOCK_IRQSAVE(sc->lock_doorbell, flags);
        sc->doorbell|=help;
        SPIN_UNLOCK_IRQRESTORE(sc->lock_doorbell, flags);
        plxwritereg(sc, L2PDBELL, help);
        handler_command|=handlercomm_doorbell;
        wakeup_handler=1;
        /*
        pERROR(sc, "doorbell_irq: 0x%08x, irqcount=%d, jiffies=%ld",
            help, sc->irqcount, jiffies);
        */
    }
    if (intcsr&plxirq_local_active) { /* local Interrupt */
        local=1;
    }
    if (intcsr&plxirq_dma0_active) { /* DMA0 Interrupt */
        pPLXIRQ(sc, "plxirq: dma0");
        SPIN_LOCK_IRQSAVE(sc->lock_intcsr, flags);
        plxwritereg(sc, INTCSR, intcsr&~plxirq_dma0);
        SPIN_UNLOCK_IRQRESTORE(sc->lock_intcsr, flags);
        /*pPLXIRQ(sc, "will call hook %p", sc->plxirq_dma0_hook);*/
        if (sc->plxirq_dma0_hook!=0) {
            sc->plxirq_dma0_hook(sc);
        } else {
            sc->got_irqs|=got_dma0;
            wakeup_local=1;
        }
    }
    if (intcsr&plxirq_dma1_active) { /* DMA1 Interrupt */
        pPLXIRQ(sc, "plxirq: dma1");
        SPIN_LOCK_IRQSAVE(sc->lock_intcsr, flags);
        plxwritereg(sc, INTCSR, intcsr&~plxirq_dma1);
        SPIN_UNLOCK_IRQRESTORE(sc->lock_intcsr, flags);
        sc->got_irqs|=got_dma1;
        wakeup_local=1;
    }
    if (local) {
        u_int32_t status;

        status=sis1100readreg(sc, sr);
        rest=status;
        pPLXIRQ(sc, "irq: status=0x%08x", status);
        if (status&irq_synch_chg) {
            rest&=~(irq_synch_chg);
            sis1100_disable_irq(sc, 0, irq_synch_chg|
                irq_reset_req|irq_prot_end|irq_prot_l_err);

            sc->got_irqs|=got_sync;
            wakeup_local=1;

            if ((status&sr_synch)==sr_synch) {
                pINFO(sc, "link is UP   status =0x%08x", status);
            } else {
                pINFO(sc, "link is DOWN status =0x%08x", status);
                sc->old_remote_hw=sc->remote_hw;
                sc->remote_hw=sis1100_hw_invalid;
                handler_command|=handlercomm_down;
                wakeup_handler=1;
            }
#ifdef __NetBSD__
            callout_reset(&sc->link_up_timer, hz, sis1100_link_up_handler, sc);
#elif __linux__
            mod_timer(&sc->link_up_timer, jiffies+HZ);
#endif
            pPLXIRQ(sc, "irq: rest=0x%08x", rest);
        }
        if (status&irq_inh_chg) {
            rest&=~(irq_inh_chg);
            pERROR(sc, "INH_CHG");
            pPLXIRQ(sc, "irq: rest=0x%08x", rest);
        }
        if (status&irq_sema_chg) {
            rest&=~(irq_sema_chg);
            pERROR(sc, "SEMA_CHG");
            pPLXIRQ(sc, "irq: rest=0x%08x", rest);
        }
        if (status&irq_rec_violation) {
            rest&=~(irq_rec_violation);
            pERROR(sc, "REC_VIOLATION");
            pPLXIRQ(sc, "irq: rest=0x%08x", rest);
        }
        if (status&irq_reset_req) {
            rest&=~(irq_reset_req);
            pERROR(sc, "RESET_REQ");
            pPLXIRQ(sc, "irq: rest=0x%08x", rest);
        }
        if (status&irq_dma_eot) {
            rest&=~(irq_dma_eot);
            pPLXIRQ(sc, "localirq: dma_eot");
            sc->got_irqs|=got_eot;
            wakeup_local=1;
            pPLXIRQ(sc, "irq: rest=0x%08x", rest);
        }
        if (status&irq_mbx0) {
            rest&=~(irq_mbx0);
            sc->mbx0=sis1100readreg(sc, mailext[0]);
            pPLXIRQ(sc, "localirq: mailext[0]=0x%x", sc->mbx0);
            handler_command|=handlercomm_mbx0;
            wakeup_handler=1;
            pPLXIRQ(sc, "irq: rest=0x%08x", rest);
        }
#ifdef NEVER
        if (status&irq_s_xoff) {
            pINFO(sc, "S_XOFF");
            pINFO(sc, "status=0x%08x", status);
            sc->got_irqs|=got_xoff;
            wakeup_local=1;
        }
#endif
        if (status&irq_lemo_in_chg) {
            rest&=~(irq_lemo_in_chg);
            pPLXIRQ(sc, "localirq: lemo_in_chg, status=0x%08x", status);
            SPIN_LOCK_IRQSAVE(sc->lock_lemo_status, flags);
            sc->lemo_status|=(status<<4)&0x30000;
            SPIN_UNLOCK_IRQRESTORE(sc->lock_lemo_status, flags);
            sc->last_opt_csr=sis1100readreg(sc, opt_csr);
            handler_command|=handlercomm_lemo;
            wakeup_handler=1;
            pPLXIRQ(sc, "irq: rest=0x%08x", rest);
        }
        if (status&irq_prot_end) {
            rest&=~(irq_prot_end);
            pPLXIRQ(sc, "localirq: got_end");
            sc->got_irqs|=got_end;
            wakeup_local=1;
            pPLXIRQ(sc, "irq: rest=0x%08x", rest);
        }
        if (status&irq_prot_l_err) {
            rest&=~(irq_prot_l_err);
            pPLXIRQ(sc, "localirq: prot_l_err");
            /*sis1100writeremreg(sc, sr, irq_prot_l_err, 0);*/
            sc->got_irqs|=got_l_err;
            wakeup_local=1;
            pPLXIRQ(sc, "irq: rest=0x%08x", rest);
        }
/*
 *         pINFO(sc, "irq: write 0x%x to sr", status);
 */
        sis1100writereg(sc, sr, status);
        pPLXIRQ(sc, "irq rest=0x%08x", rest);
    }

    if (wakeup_local)
        wakeup(&sc->local_wait);

    if (wakeup_handler) {
        /*pERROR(sc, "sis1100_intr: handler_command=0x%x", handler_command);*/
        SPIN_LOCK_IRQSAVE(sc->handlercommand.lock, flags);
        sc->handlercommand.command=handler_command;
        SPIN_UNLOCK_IRQRESTORE(sc->handlercommand.lock, flags);
        wakeup(&sc->handler_wait);
    }

    return IRQ_HANDLED;
}
